import Link from "next/link"
import Image from "next/image"
import { Card, CardContent, CardFooter } from "@/components/ui/card"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Badge } from "@/components/ui/badge"
import { Eye, Heart } from "lucide-react"
import type { Post } from "@/lib/mock-data"

interface PostCardProps {
  post: Post
  featured?: boolean
}

export function PostCard({ post, featured = false }: PostCardProps) {
  const formattedDate = new Date(post.publishedAt).toLocaleDateString("en-US", {
    month: "short",
    day: "numeric",
    year: "numeric",
  })

  if (featured) {
    return (
      <Link href={`/post/${post.slug}`}>
        <Card className="overflow-hidden hover:shadow-lg transition-shadow group">
          <div className="grid md:grid-cols-2 gap-0">
            <div className="relative h-64 md:h-full">
              <Image
                src={post.featuredImage || "/placeholder.svg"}
                alt={post.title}
                fill
                sizes="(min-width: 768px) 50vw, 100vw"
                priority
                className="object-cover group-hover:scale-105 transition-transform duration-300"
              />
            </div>
            <CardContent className="p-6 md:p-8 flex flex-col justify-center">
              <Badge className="w-fit mb-4">{post.categoryName}</Badge>
              <h2 className="text-2xl md:text-3xl font-bold mb-3 group-hover:text-primary transition-colors text-balance">
                {post.title}
              </h2>
              <p className="text-muted-foreground mb-6 line-clamp-2 leading-relaxed">{post.excerpt}</p>
              <div className="flex items-center gap-4">
                <div className="flex items-center gap-2">
                  <Avatar className="h-10 w-10">
                    <AvatarImage src={post.authorAvatar || "/placeholder.svg"} alt={post.authorName} />
                    <AvatarFallback>{post.authorName.charAt(0)}</AvatarFallback>
                  </Avatar>
                  <div>
                    <p className="text-sm font-medium">{post.authorName}</p>
                    <p className="text-xs text-muted-foreground">{formattedDate}</p>
                  </div>
                </div>
                <div className="ml-auto flex items-center gap-4 text-sm text-muted-foreground">
                  <span className="flex items-center gap-1">
                    <Eye className="h-4 w-4" />
                    {post.views}
                  </span>
                  <span className="flex items-center gap-1">
                    <Heart className="h-4 w-4" />
                    {post.likes}
                  </span>
                </div>
              </div>
            </CardContent>
          </div>
        </Card>
      </Link>
    )
  }

  return (
    <Link href={`/post/${post.slug}`}>
      <Card className="overflow-hidden hover:shadow-lg transition-shadow group h-full flex flex-col">
        <div className="relative h-48 overflow-hidden">
          <Image
            src={post.featuredImage || "/placeholder.svg"}
            alt={post.title}
            fill
            sizes="(min-width: 1024px) 33vw, (min-width: 768px) 50vw, 100vw"
            className="object-cover group-hover:scale-105 transition-transform duration-300"
          />
        </div>
        <CardContent className="p-6 flex-1 flex flex-col">
          <Badge className="w-fit mb-3">{post.categoryName}</Badge>
          <h3 className="text-xl font-bold mb-2 group-hover:text-primary transition-colors line-clamp-2 text-balance">
            {post.title}
          </h3>
          <p className="text-muted-foreground text-sm mb-4 line-clamp-2 leading-relaxed flex-1">{post.excerpt}</p>
          <div className="flex items-center justify-between pt-4 border-t">
            <div className="flex items-center gap-2">
              <Avatar className="h-8 w-8">
                <AvatarImage src={post.authorAvatar || "/placeholder.svg"} alt={post.authorName} />
                <AvatarFallback>{post.authorName.charAt(0)}</AvatarFallback>
              </Avatar>
              <div>
                <p className="text-sm font-medium">{post.authorName}</p>
                <p className="text-xs text-muted-foreground">{formattedDate}</p>
              </div>
            </div>
          </div>
        </CardContent>
        <CardFooter className="px-6 pb-6 pt-0 flex items-center gap-4 text-sm text-muted-foreground">
          <span className="flex items-center gap-1">
            <Eye className="h-4 w-4" />
            {post.views}
          </span>
          <span className="flex items-center gap-1">
            <Heart className="h-4 w-4" />
            {post.likes}
          </span>
        </CardFooter>
      </Card>
    </Link>
  )
}
