"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { CommentItem } from "./comment-item"
import { useAuth } from "@/lib/auth-context"
import { mockComments, type Comment } from "@/lib/mock-data"
import { MessageCircle } from "lucide-react"
import Link from "next/link"

interface CommentsSectionProps {
  postId: string
}

export function CommentsSection({ postId }: CommentsSectionProps) {
  const { user } = useAuth()
  const [comments, setComments] = useState<Comment[]>(mockComments.filter((c) => c.postId === postId))
  const [newComment, setNewComment] = useState("")
  const [isSubmitting, setIsSubmitting] = useState(false)

  const handleSubmitComment = async () => {
    if (!newComment.trim() || !user) return

    setIsSubmitting(true)

    // Simulate API call
    await new Promise((resolve) => setTimeout(resolve, 500))

    const comment: Comment = {
      id: Math.random().toString(36).substr(2, 9),
      postId,
      userId: user.id,
      userName: user.fullName,
      userAvatar: user.avatarUrl || "/placeholder.svg",
      content: newComment,
      createdAt: new Date().toISOString(),
    }

    setComments([comment, ...comments])
    setNewComment("")
    setIsSubmitting(false)
  }

  const handleReply = async (commentId: string, content: string) => {
    if (!user) return

    // Simulate API call
    await new Promise((resolve) => setTimeout(resolve, 500))

    const reply: Comment = {
      id: Math.random().toString(36).substr(2, 9),
      postId,
      userId: user.id,
      userName: user.fullName,
      userAvatar: user.avatarUrl || "/placeholder.svg",
      content,
      parentId: commentId,
      createdAt: new Date().toISOString(),
    }

    setComments((prevComments) => {
      const addReplyToComment = (comments: Comment[]): Comment[] => {
        return comments.map((comment) => {
          if (comment.id === commentId) {
            return {
              ...comment,
              replies: [...(comment.replies || []), reply],
            }
          }
          if (comment.replies) {
            return {
              ...comment,
              replies: addReplyToComment(comment.replies),
            }
          }
          return comment
        })
      }
      return addReplyToComment(prevComments)
    })
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <MessageCircle className="h-5 w-5" />
          Comments ({comments.length})
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {user ? (
          <div className="space-y-3">
            <Textarea
              placeholder="Share your thoughts..."
              value={newComment}
              onChange={(e) => setNewComment(e.target.value)}
              rows={4}
            />
            <Button onClick={handleSubmitComment} disabled={isSubmitting || !newComment.trim()}>
              {isSubmitting ? "Posting..." : "Post Comment"}
            </Button>
          </div>
        ) : (
          <div className="text-center py-6 bg-muted/30 rounded-lg">
            <p className="text-muted-foreground mb-4">Sign in to join the conversation</p>
            <Link href="/auth">
              <Button>Sign In</Button>
            </Link>
          </div>
        )}

        <div className="space-y-6 pt-6 border-t">
          {comments.length === 0 ? (
            <p className="text-center text-muted-foreground py-8">
              No comments yet. Be the first to share your thoughts!
            </p>
          ) : (
            comments.map((comment) => <CommentItem key={comment.id} comment={comment} onReply={handleReply} />)
          )}
        </div>
      </CardContent>
    </Card>
  )
}
