import type { Metadata } from "next"
import { Header } from "@/components/layout/header"
import { Footer } from "@/components/layout/footer"
import { PostCard } from "@/components/blog/post-card"
import { CategoryFilterLinks } from "@/components/blog/category-filter-links"
import { Button } from "@/components/ui/button"
import { mockPosts } from "@/lib/mock-data"
import { TrendingUp } from "lucide-react"

export const metadata: Metadata = {
  title: "Insight - Modern Blogging Platform",
}

type PageProps = {
  searchParams?: Record<string, string | string[] | undefined>
}

export default function HomePage({ searchParams = {} }: PageProps) {
  const selectedCategoryParam = Array.isArray(searchParams.category) ? searchParams.category[0] : searchParams.category
  const selectedCategory = selectedCategoryParam ?? null

  const filteredPosts = selectedCategory
    ? mockPosts.filter((post) => post.categoryId === selectedCategory)
    : mockPosts

  const featuredPost = mockPosts[0]
  const regularPosts = filteredPosts.slice(1)

  return (
    <div className="min-h-screen flex flex-col">
      <Header />

      <main className="flex-1">
        {/* Hero Section */}
        <section className="bg-gradient-to-br from-primary/5 via-accent/5 to-background py-16 md:py-24">
          <div className="container mx-auto px-4">
            <div className="max-w-3xl mx-auto text-center space-y-6">
              <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-primary/10 text-primary text-sm font-medium">
                <TrendingUp className="h-4 w-4" />
                Welcome to Insight
              </div>
              <h1 className="text-4xl md:text-6xl font-bold text-balance">
                Discover Stories That <span className="text-primary">Inspire</span>
              </h1>
              <p className="text-lg md:text-xl text-muted-foreground leading-relaxed text-pretty">
                Join our community of writers and readers sharing insights on technology, design, business, and
                lifestyle.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center pt-4">
                <Button size="lg" className="text-base">
                  Start Reading
                </Button>
                <Button size="lg" variant="outline" className="text-base bg-transparent">
                  Become a Writer
                </Button>
              </div>
            </div>
          </div>
        </section>

        {/* Featured Post */}
        {!selectedCategory && (
          <section className="container mx-auto px-4 py-12 md:py-16">
            <div className="flex items-center gap-2 mb-6">
              <div className="h-1 w-12 bg-primary rounded-full" />
              <h2 className="text-2xl font-bold">Featured Story</h2>
            </div>
            <PostCard post={featuredPost} featured />
          </section>
        )}

        {/* Category Filter */}
        <section className="container mx-auto px-4 py-8">
          <CategoryFilterLinks selectedCategory={selectedCategory} />
        </section>

        {/* Blog Posts Grid */}
        <section className="container mx-auto px-4 pb-16">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {regularPosts.map((post) => (
              <PostCard key={post.id} post={post} />
            ))}
          </div>

          {regularPosts.length === 0 && (
            <div className="text-center py-12">
              <p className="text-muted-foreground">No posts found in this category.</p>
            </div>
          )}

          {regularPosts.length > 0 && (
            <div className="flex justify-center mt-12">
              <Button variant="outline" size="lg">
                Load More Posts
              </Button>
            </div>
          )}
        </section>

        {/* Newsletter Section */}
        <section className="bg-muted/30 py-16">
          <div className="container mx-auto px-4">
            <div className="max-w-2xl mx-auto text-center space-y-6">
              <h2 className="text-3xl md:text-4xl font-bold text-balance">Stay Updated</h2>
              <p className="text-muted-foreground leading-relaxed">
                Get the latest stories delivered straight to your inbox. Join thousands of readers.
              </p>
              <div className="flex flex-col sm:flex-row gap-3 max-w-md mx-auto">
                <input
                  type="email"
                  placeholder="Enter your email"
                  className="flex-1 px-4 py-3 rounded-lg border bg-background focus:outline-none focus:ring-2 focus:ring-primary"
                />
                <Button size="lg">Subscribe</Button>
              </div>
            </div>
          </div>
        </section>
      </main>

      <Footer />
    </div>
  )
}
