"use client"

import { useEffect, useState } from "react"
import { useRouter } from "next/navigation"
import Link from "next/link"
import { Header } from "@/components/layout/header"
import { Footer } from "@/components/layout/footer"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { StatsCard } from "@/components/admin/stats-card"
import { useAuth } from "@/lib/auth-context"
import { mockPosts, mockCategories } from "@/lib/mock-data"
import { FileText, Users, Eye, TrendingUp, Edit, Trash2, MoreVertical, CheckCircle, XCircle, Clock } from "lucide-react"
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { useToast } from "@/hooks/use-toast"

export default function AdminDashboard() {
  const router = useRouter()
  const { user } = useAuth()
  const { toast } = useToast()
  const [posts, setPosts] = useState(mockPosts)

  useEffect(() => {
    if (!user || user.role !== "admin") {
      router.push("/")
      toast({
        title: "Access denied",
        description: "You don't have permission to access the admin dashboard.",
        variant: "destructive",
      })
    }
  }, [user, router, toast])

  if (!user || user.role !== "admin") {
    return null
  }

  const totalPosts = posts.length
  const publishedPosts = posts.filter((p) => p.status === "published").length
  const draftPosts = posts.filter((p) => p.status === "draft").length
  const totalViews = posts.reduce((sum, post) => sum + post.views, 0)
  const totalUsers = 156 // Mock data

  const handleDeletePost = (postId: string) => {
    setPosts(posts.filter((p) => p.id !== postId))
    toast({
      title: "Post deleted",
      description: "The post has been successfully deleted.",
    })
  }

  const handleStatusChange = (postId: string, newStatus: "published" | "draft" | "archived") => {
    setPosts(posts.map((p) => (p.id === postId ? { ...p, status: newStatus } : p)))
    toast({
      title: "Status updated",
      description: `Post status changed to ${newStatus}.`,
    })
  }

  return (
    <div className="min-h-screen flex flex-col">
      <Header />

      <main className="flex-1 container mx-auto px-4 py-8">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">Admin Dashboard</h1>
          <p className="text-muted-foreground">Manage your blog posts, users, and content</p>
        </div>

        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <StatsCard
            title="Total Posts"
            value={totalPosts}
            description={`${publishedPosts} published, ${draftPosts} drafts`}
            icon={FileText}
            trend={{ value: 12, isPositive: true }}
          />
          <StatsCard
            title="Total Views"
            value={totalViews.toLocaleString()}
            description="Across all posts"
            icon={Eye}
            trend={{ value: 8, isPositive: true }}
          />
          <StatsCard
            title="Total Users"
            value={totalUsers}
            description="Registered members"
            icon={Users}
            trend={{ value: 15, isPositive: true }}
          />
          <StatsCard
            title="Engagement"
            value="4.2%"
            description="Average engagement rate"
            icon={TrendingUp}
            trend={{ value: 2.3, isPositive: true }}
          />
        </div>

        {/* Recent Posts Table */}
        <Card className="mb-8">
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle>All Posts</CardTitle>
                <CardDescription>Manage and moderate blog posts</CardDescription>
              </div>
              <Link href="/editor/new">
                <Button>Create New Post</Button>
              </Link>
            </div>
          </CardHeader>
          <CardContent>
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Title</TableHead>
                  <TableHead>Author</TableHead>
                  <TableHead>Category</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Views</TableHead>
                  <TableHead>Date</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {posts.slice(0, 10).map((post) => (
                  <TableRow key={post.id}>
                    <TableCell className="font-medium max-w-xs truncate">{post.title}</TableCell>
                    <TableCell>{post.authorName}</TableCell>
                    <TableCell>
                      <Badge variant="outline">{post.categoryName}</Badge>
                    </TableCell>
                    <TableCell>
                      {post.status === "published" && (
                        <Badge className="gap-1">
                          <CheckCircle className="h-3 w-3" />
                          Published
                        </Badge>
                      )}
                      {post.status === "draft" && (
                        <Badge variant="secondary" className="gap-1">
                          <Clock className="h-3 w-3" />
                          Draft
                        </Badge>
                      )}
                      {post.status === "archived" && (
                        <Badge variant="outline" className="gap-1">
                          <XCircle className="h-3 w-3" />
                          Archived
                        </Badge>
                      )}
                    </TableCell>
                    <TableCell>{post.views.toLocaleString()}</TableCell>
                    <TableCell>{new Date(post.publishedAt).toLocaleDateString()}</TableCell>
                    <TableCell className="text-right">
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button variant="ghost" size="icon">
                            <MoreVertical className="h-4 w-4" />
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuItem asChild>
                            <Link href={`/post/${post.slug}`}>View Post</Link>
                          </DropdownMenuItem>
                          <DropdownMenuItem asChild>
                            <Link href={`/editor/${post.id}`}>
                              <Edit className="h-4 w-4 mr-2" />
                              Edit
                            </Link>
                          </DropdownMenuItem>
                          {post.status !== "published" && (
                            <DropdownMenuItem onClick={() => handleStatusChange(post.id, "published")}>
                              <CheckCircle className="h-4 w-4 mr-2" />
                              Publish
                            </DropdownMenuItem>
                          )}
                          {post.status !== "draft" && (
                            <DropdownMenuItem onClick={() => handleStatusChange(post.id, "draft")}>
                              <Clock className="h-4 w-4 mr-2" />
                              Move to Draft
                            </DropdownMenuItem>
                          )}
                          {post.status !== "archived" && (
                            <DropdownMenuItem onClick={() => handleStatusChange(post.id, "archived")}>
                              <XCircle className="h-4 w-4 mr-2" />
                              Archive
                            </DropdownMenuItem>
                          )}
                          <DropdownMenuItem
                            onClick={() => handleDeletePost(post.id)}
                            className="text-destructive focus:text-destructive"
                          >
                            <Trash2 className="h-4 w-4 mr-2" />
                            Delete
                          </DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </CardContent>
        </Card>

        {/* Categories Overview */}
        <Card>
          <CardHeader>
            <CardTitle>Categories</CardTitle>
            <CardDescription>Content distribution across categories</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              {mockCategories.map((category) => (
                <Card key={category.id}>
                  <CardContent className="p-4">
                    <h3 className="font-semibold mb-1">{category.name}</h3>
                    <p className="text-2xl font-bold text-primary mb-1">{category.postCount}</p>
                    <p className="text-xs text-muted-foreground">posts</p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </CardContent>
        </Card>
      </main>

      <Footer />
    </div>
  )
}
